<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Proteção da página: Apenas administradores podem acessar
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || $_SESSION['user_perfil'] !== 'admin') {
    header('Location: login.html?message=' . urlencode('Acesso negado. Apenas administradores podem criar contas.') . '&type=error');
    exit();
}

// Inclua a conexão com o banco de dados
require_once 'db.php';

$message = '';
$type = '';

// Função de validação
function validarEntrada($nome, $email, $senha) {
    $erros = [];

    // Validação do nome
    if (empty($nome) || strlen($nome) < 3) {
        $erros[] = "Nome deve ter no mínimo 3 caracteres";
    }

    // Validação do email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $erros[] = "E-mail inválido";
    }

    // Validação da senha
    if (strlen($senha) < 6) {
        $erros[] = "Senha deve ter no mínimo 6 caracteres";
    }

    return $erros;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nome = trim($_POST['nome'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $senha_clara = $_POST['senha'] ?? '';
    $perfil = $_POST['perfil'] ?? 'cliente';

    // Validação dos dados
    $errosValidacao = validarEntrada($nome, $email, $senha_clara);

    if (!empty($errosValidacao)) {
        $message = implode("\n", $errosValidacao);
        $type = 'error';
    } else {
        try {
            // Verifica se o email já existe
            $stmt_check = $pdo->prepare('SELECT id FROM usuarios WHERE email = :email');
            $stmt_check->execute(['email' => $email]);
            
            if ($stmt_check->fetch()) {
                $message = 'Este e-mail já está cadastrado.';
                $type = 'error';
            } else {
                // Crie o hash da senha
                $senha_hash = password_hash($senha_clara, PASSWORD_DEFAULT);

                // Insere o novo usuário com status ativo
                $stmt_insert = $pdo->prepare('INSERT INTO usuarios (nome, email, senha, perfil, status) VALUES (:nome, :email, :senha, :perfil, :status)');
                $resultado = $stmt_insert->execute([
                    'nome' => $nome,
                    'email' => $email,
                    'senha' => $senha_hash,
                    'perfil' => $perfil,
                    'status' => 'ativo'
                ]);

                if ($resultado) {
                    $message = 'Nova conta criada com sucesso!';
                    $type = 'success';
                    
                    // Log de criação de conta (opcional)
                    error_log("Nova conta criada: {$email} - Perfil: {$perfil}");
                } else {
                    $message = 'Erro ao criar conta. Tente novamente.';
                    $type = 'error';
                }
            }
        } catch (PDOException $e) {
            // Log detalhado do erro
            error_log("Erro PDO: " . $e->getMessage());
            $message = 'Erro interno do sistema. Contate o suporte.';
            $type = 'error';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Criar Nova Conta</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-blue: #0066ff;
            --secondary-blue: #1e40af;
            --accent-blue: #3b82f6;
            --light-blue: #60a5fa;
            --dark-blue: #1e3a8a;
            --cyan: #06b6d4;
            --bg-dark: #0f1419;
            --bg-card: #1a1f2e;
            --bg-glass: rgba(26, 31, 46, 0.85);
            --text-primary: #ffffff;
            --text-secondary: #94a3b8;
            --shadow-primary: 0 8px 32px rgba(0, 102, 255, 0.25);
            --shadow-glow: 0 0 40px rgba(0, 102, 255, 0.4);
            --gradient-primary: linear-gradient(135deg, var(--primary-blue) 0%, var(--cyan) 100%);
            --gradient-bg: linear-gradient(135deg, #0f1419 0%, #1e3a8a 100%);
            --radius: 24px;
            --radius-sm: 16px;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: var(--gradient-bg);
            color: var(--text-primary);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
            overflow: hidden;
            position: relative;
        }

        /* Animated background particles */
        .bg-animation {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 0;
            overflow: hidden;
        }

        .particle {
            position: absolute;
            background: var(--primary-blue);
            border-radius: 50%;
            animation: float 6s ease-in-out infinite;
            opacity: 0.1;
        }

        .particle:nth-child(1) {
            width: 8px;
            height: 8px;
            top: 20%;
            left: 10%;
            animation-delay: 0s;
        }

        .particle:nth-child(2) {
            width: 12px;
            height: 12px;
            top: 60%;
            left: 80%;
            animation-delay: 2s;
        }

        .particle:nth-child(3) {
            width: 6px;
            height: 6px;
            top: 40%;
            left: 60%;
            animation-delay: 4s;
        }

        .particle:nth-child(4) {
            width: 10px;
            height: 10px;
            top: 80%;
            left: 20%;
            animation-delay: 1s;
        }

        .particle:nth-child(5) {
            width: 14px;
            height: 14px;
            top: 30%;
            left: 90%;
            animation-delay: 3s;
        }

        @keyframes float {
            0%, 100% {
                transform: translateY(0px) rotate(0deg);
            }
            33% {
                transform: translateY(-20px) rotate(120deg);
            }
            66% {
                transform: translateY(10px) rotate(240deg);
            }
        }

        .container {
            background: var(--bg-glass);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(59, 130, 246, 0.3);
            border-radius: var(--radius);
            box-shadow: var(--shadow-primary);
            width: 100%;
            max-width: 500px;
            padding: 3rem;
            position: relative;
            overflow: hidden;
            z-index: 10;
            animation: slideUp 0.8s ease-out;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: var(--gradient-primary);
            animation: shimmer 2s infinite;
        }

        @keyframes shimmer {
            0% {
                opacity: 0.5;
            }
            100% {
                opacity: 1;
            }
        }

        .header {
            text-align: center;
            margin-bottom: 2.5rem;
        }

        .header h1 {
            font-size: 2.5rem;
            font-weight: 900;
            background: var(--gradient-primary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 1rem;
            letter-spacing: -0.02em;
            line-height: 1.1;
            animation: titleGlow 3s ease-in-out infinite alternate;
        }

        @keyframes titleGlow {
            from {
                filter: drop-shadow(0 0 20px rgba(0, 102, 255, 0.3));
            }
            to {
                filter: drop-shadow(0 0 40px rgba(6, 182, 212, 0.5));
            }
        }

        .header p {
            color: var(--text-secondary);
            font-size: 1.1rem;
            font-weight: 400;
        }

        .message {
            padding: 1.25rem 1.5rem;
            margin-bottom: 2rem;
            border-radius: var(--radius-sm);
            font-size: 1rem;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 12px;
            backdrop-filter: blur(10px);
            border: 2px solid transparent;
            animation: messageSlide 0.5s ease-out;
        }

        @keyframes messageSlide {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .message.error {
            background: rgba(239, 68, 68, 0.15);
            border-color: rgba(239, 68, 68, 0.4);
            color: #fca5a5;
            box-shadow: 0 4px 20px rgba(239, 68, 68, 0.2);
        }

        .message.success {
            background: rgba(16, 185, 129, 0.15);
            border-color: rgba(16, 185, 129, 0.4);
            color: #6ee7b7;
            box-shadow: 0 4px 20px rgba(16, 185, 129, 0.2);
        }

        .form-group {
            margin-bottom: 2rem;
            position: relative;
        }

        .form-group label {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 0.75rem;
            color: var(--text-primary);
            font-weight: 600;
            font-size: 1rem;
        }

        .form-group label svg {
            width: 20px;
            height: 20px;
            stroke: var(--primary-blue);
            fill: none;
        }

        .input-wrapper {
            position: relative;
        }

        .form-group input,
        .form-group select {
            width: 100%;
            height: 60px;
            background: var(--bg-card);
            border: 2px solid var(--secondary-blue);
            border-radius: var(--radius-sm);
            padding: 0 1.5rem;
            font-size: 1.1rem;
            color: var(--text-primary);
            font-family: inherit;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            backdrop-filter: blur(10px);
        }

        .form-group input::placeholder {
            color: var(--text-secondary);
        }

        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--primary-blue);
            box-shadow: 0 0 0 4px rgba(0, 102, 255, 0.1);
            transform: translateY(-2px);
        }

        .form-group select {
            cursor: pointer;
            appearance: none;
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%2394a3b8' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
            background-position: right 1.5rem center;
            background-repeat: no-repeat;
            background-size: 16px;
            padding-right: 3.5rem;
        }

        .form-group input:hover,
        .form-group select:hover {
            border-color: var(--light-blue);
        }

        .btn-submit {
            width: 100%;
            height: 60px;
            background: var(--gradient-primary);
            color: white;
            border: none;
            border-radius: var(--radius-sm);
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-top: 1rem;
            box-shadow: var(--shadow-primary);
            position: relative;
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.75rem;
        }

        .btn-submit::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
            transition: left 0.5s;
        }

        .btn-submit:hover::before {
            left: 100%;
        }

        .btn-submit:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-glow);
        }

        .btn-submit:active {
            transform: translateY(0);
        }

        .btn-submit:disabled {
            opacity: 0.7;
            cursor: not-allowed;
            transform: none;
        }

        .back-link {
            text-align: center;
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid rgba(59, 130, 246, 0.3);
        }

        .back-link a {
            color: var(--primary-blue);
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            display: inline-flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1.5rem;
            border-radius: var(--radius-sm);
            background: rgba(0, 102, 255, 0.1);
            border: 1px solid rgba(0, 102, 255, 0.3);
            backdrop-filter: blur(10px);
        }

        .back-link a:hover {
            color: var(--cyan);
            transform: translateX(-3px);
            background: rgba(0, 102, 255, 0.2);
            border-color: var(--primary-blue);
            box-shadow: 0 4px 15px rgba(0, 102, 255, 0.2);
        }

        .back-link a svg {
            width: 18px;
            height: 18px;
            stroke: currentColor;
            fill: none;
        }

        /* Responsividade */
        @media (max-width: 768px) {
            .container {
                padding: 2rem 1.5rem;
                margin: 1rem;
            }

            .header h1 {
                font-size: 2rem;
            }

            .form-group input,
            .form-group select,
            .btn-submit {
                height: 56px;
                font-size: 1rem;
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 1.5rem 1rem;
            }

            .header h1 {
                font-size: 1.75rem;
            }

            .form-group input,
            .form-group select,
            .btn-submit {
                height: 52px;
                font-size: 0.95rem;
            }
        }
    </style>
</head>
<body>
    <!-- Animated background particles -->
    <div class="bg-animation">
        <div class="particle"></div>
        <div class="particle"></div>
        <div class="particle"></div>
        <div class="particle"></div>
        <div class="particle"></div>
    </div>

    <div class="container">
        <div class="header">
            <h1>
                <svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="display: inline-block; vertical-align: middle; margin-right: 0.5rem;">
                    <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                    <circle cx="9" cy="7" r="4"/>
                    <line x1="19" y1="8" x2="19" y2="14"/>
                    <line x1="22" y1="11" x2="16" y2="11"/>
                </svg>
                Criar Nova Conta
            </h1>
            <p>Adicione um novo usuário ao sistema</p>
        </div>

        <?php if ($message): ?>
            <div class="message <?php echo $type; ?>">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <?php if ($type === 'error'): ?>
                        <path d="m21.73 18-8-14a2 2 0 0 0-3.46 0l-8 14A2 2 0 0 0 4 21h16a2 2 0 0 0 1.73-3Z"/>
                        <path d="M12 9v4"/>
                        <path d="m12 17 .01 0"/>
                    <?php else: ?>
                        <path d="M12 2a10 10 0 1 0 10 10A10 10 0 0 0 12 2Z"/>
                        <path d="m9 12 2 2 4-4"/>
                    <?php endif; ?>
                </svg>
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <form action="criar_conta.php" method="POST">
            <div class="form-group">
                <label for="nome">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                        <circle cx="12" cy="7" r="4"/>
                    </svg>
                    Nome Completo
                </label>
                <div class="input-wrapper">
                    <input type="text" id="nome" name="nome" required placeholder="Digite o nome completo">
                </div>
            </div>

            <div class="form-group">
                <label for="email">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"/>
                        <polyline points="22,6 12,13 2,6"/>
                    </svg>
                    E-mail
                </label>
                <div class="input-wrapper">
                    <input type="email" id="email" name="email" required placeholder="Digite o e-mail">
                </div>
            </div>

            <div class="form-group">
                <label for="senha">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                        <circle cx="12" cy="16" r="1"/>
                        <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                    </svg>
                    Senha
                </label>
                <div class="input-wrapper">
                    <input type="password" id="senha" name="senha" required placeholder="Digite a senha">
                </div>
            </div>

            <div class="form-group">
                <label for="perfil">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                        <circle cx="9" cy="7" r="4"/>
                        <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                        <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                    </svg>
                    Perfil de Usuário
                </label>
                <select id="perfil" name="perfil">
                    <option value="cliente">Cliente</option>
                    <option value="admin">Administrador</option>
                </select>
            </div>

            <button type="submit" class="btn-submit">
                <svg width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="10"/>
                    <line x1="12" y1="8" x2="12" y2="16"/>
                    <line x1="8" y1="12" x2="16" y2="12"/>
                </svg>
                <span class="btn-text">Criar Conta</span>
            </button>
        </form>

        <div class="back-link">
            <a href="dashboard.php">
                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="19" y1="12" x2="5" y2="12"/>
                    <polyline points="12,19 5,12 12,5"/>
                </svg>
                Voltar ao Dashboard
            </a>
        </div>
    </div>

    <script>
        // Adiciona efeitos visuais aos inputs
        document.querySelectorAll('input, select').forEach(input => {
            input.addEventListener('focus', function() {
                this.parentElement.parentElement.style.transform = 'scale(1.02)';
                this.parentElement.parentElement.style.transition = 'all 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
            });
            
            input.addEventListener('blur', function() {
                this.parentElement.parentElement.style.transform = 'scale(1)';
            });
        });

        // Validação em tempo real do email
        document.getElementById('email').addEventListener('input', function() {
            const email = this.value;
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            
            if (email && !emailRegex.test(email)) {
                this.style.borderColor = '#ef4444';
                this.style.boxShadow = '0 0 0 4px rgba(239, 68, 68, 0.1)';
            } else {
                this.style.borderColor = '';
                this.style.boxShadow = '';
            }
        });

        // Validação da força da senha
        document.getElementById('senha').addEventListener('input', function() {
            const senha = this.value;
            const strongPassword = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/;
            
            if (senha.length > 0 && senha.length < 6) {
                this.style.borderColor = '#ef4444';
                this.style.boxShadow = '0 0 0 4px rgba(239, 68, 68, 0.1)';
            } else if (senha.length >= 6 && !strongPassword.test(senha)) {
                this.style.borderColor = '#f59e0b';
                this.style.boxShadow = '0 0 0 4px rgba(245, 158, 11, 0.1)';
            } else if (strongPassword.test(senha)) {
                this.style.borderColor = '#10b981';
                this.style.boxShadow = '0 0 0 4px rgba(16, 185, 129, 0.1)';
            } else {
                this.style.borderColor = '';
                this.style.boxShadow = '';
            }
        });

        // Animação do botão submit
        document.querySelector('.btn-submit').addEventListener('click', function(e) {
            // Verificação básica antes do envio
            const nome = document.getElementById('nome').value;
            const email = document.getElementById('email').value;
            const senha = document.getElementById('senha').value;
            
            if (!nome || !email || !senha) {
                e.preventDefault();
                return;
            }
            
            const btnText = this.querySelector('.btn-text');
            const originalText = btnText.textContent;
            
            btnText.innerHTML = '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="animation: spin 1s linear infinite;"><circle cx="12" cy="12" r="3"/><path d="m12 1 0 6"/><path d="m12 17 0 6"/><path d="m4.22 4.22 4.24 4.24"/><path d="m15.54 15.54 4.24 4.24"/><path d="m1 12 6 0"/><path d="m17 12 6 0"/><path d="m4.22 19.78 4.24-4.24"/><path d="m15.54 8.46 4.24-4.24"/></svg> Criando...';
            this.disabled = true;
            
            // Re-habilita o botão após 5 segundos (caso não haja redirecionamento)
            setTimeout(() => {
                btnText.textContent = originalText;
                this.disabled = false;
            }, 5000);
        });

        // Animação de entrada suave para os elementos
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.style.animationPlayState = 'running';
                }
            });
        });

        document.querySelectorAll('.form-group').forEach(group => {
            observer.observe(group);
        });

        // Adiciona um efeito de partículas no fundo
        function createParticle() {
            const particle = document.createElement('div');
            particle.className = 'particle';
            particle.style.left = Math.random() * 100 + '%';
            particle.style.top = Math.random() * 100 + '%';
            particle.style.width = (Math.random() * 10 + 5) + 'px';
            particle.style.height = particle.style.width;
            particle.style.animationDelay = Math.random() * 6 + 's';
            
            document.querySelector('.bg-animation').appendChild(particle);
            
            setTimeout(() => {
                particle.remove();
            }, 6000);
        }

        // Cria novas partículas periodicamente
        setInterval(createParticle, 2000);
    </script>

    <style>
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</body>
</html>